<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.monetumoads
 *
 * @copyright   Copyright (C) 2025 Monetumo. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\Monetumoads\Extension;

defined('_JEXEC') or die;

use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Uri\Uri;
use Joomla\Event\SubscriberInterface;

/**
 * Monetumo Ad Monetization Plugin
 *
 * Integrates Monetumo ad monetization solution with Joomla sites by:
 * - Injecting required scripts and CSS
 * - Adding resource hints for performance
 * - Managing ads.txt file delivery
 *
 * @since  1.0.0
 */
final class Monetumo extends CMSPlugin implements SubscriberInterface
{
  /**
   * Load the language file on instantiation.
   *
   * @var    boolean
   * @since  1.0.0
   */
  protected $autoloadLanguage = true;

  /**
   * Returns an array of events this subscriber will listen to.
   *
   * @return  array
   *
   * @since   1.0.0
   */
  public static function getSubscribedEvents(): array
  {
    return [
      'onBeforeCompileHead' => 'onBeforeCompileHead',
      'onAfterInitialise'   => 'onAfterInitialise',
    ];
  }

  /**
   * Injects Monetumo scripts, CSS, and resource hints into the document head.
   *
   * @return  void
   *
   * @since   1.0.0
   */
  public function onBeforeCompileHead(): void
  {
    $app = $this->getApplication();

    // Only run on frontend
    if (!$app->isClient('site')) {
      return;
    }

    $bundleId = $this->params->get('bundle_id', '');

    // Exit if bundle ID is not configured
    if (empty($bundleId)) {
      return;
    }

    $document = $app->getDocument();
    $wa = $document->getWebAssetManager();

    // Sanitize bundle ID for use in URLs
    $safeBundleId = $this->sanitizeBundleId($bundleId);

    // Register and add CMP script with data-cfasync attribute
    $wa->registerScript(
      'monetumo.cmp',
      'https://b-cdn.monetumo.com/cmp/' . $safeBundleId . '.js',
      [],
      ['data-cfasync' => 'false'],
      []
    );
    $wa->useScript('monetumo.cmp');

    // Register and add GPT script with async attribute
    $wa->registerScript(
      'monetumo.gpt',
      'https://securepubads.g.doubleclick.net/tag/js/gpt.js',
      [],
      ['async' => true],
      []
    );
    $wa->useScript('monetumo.gpt');

    // Register and add Bundle script with defer and data-cfasync attributes
    $wa->registerScript(
      'monetumo.bundle',
      'https://b-cdn.monetumo.com/bundles/' . $safeBundleId . '.js',
      [],
      ['defer' => true, 'data-cfasync' => 'false'],
      []
    );
    $wa->useScript('monetumo.bundle');

    // Register and add CSS
    $wa->registerStyle(
      'monetumo.style',
      'https://b-cdn.monetumo.com/cls-css/' . $safeBundleId . '.css',
      [],
      [],
      []
    );
    $wa->useStyle('monetumo.style');

    // Add resource hints for performance
    $this->addResourceHints($document);
  }

  /**
   * Adds preconnect resource hints for performance.
   *
   * @param   object  $document  The document object.
   *
   * @return  void
   *
   * @since   1.0.0
   */
  private function addResourceHints($document): void
  {
    $domains = [
      'https://securepubads.g.doubleclick.net',
      'https://b-cdn.monetumo.com',
    ];

    foreach ($domains as $domain) {
      // Add preconnect with crossorigin for CORS requests
      $document->addHeadLink(
        $domain,
        'preconnect',
        'rel',
        ['crossorigin' => 'anonymous']
      );
    }
  }

  /**
   * Intercepts ads.txt requests and serves content based on strategy.
   *
   * @return  void
   *
   * @since   1.0.0
   */
  public function onAfterInitialise(): void
  {
    $app = $this->getApplication();

    // Only run on frontend
    if (!$app->isClient('site')) {
      return;
    }

    // Check if this is an ads.txt request
    $uri = Uri::getInstance();
    $path = trim($uri->getPath(), '/');

    if ($path !== 'ads.txt') {
      return;
    }

    $bundleId = $this->params->get('bundle_id', '');

    if (empty($bundleId)) {
      $this->sendPlainTextResponse(
        Text::_('PLG_SYSTEM_MONETUMOADS_ERROR_NO_BUNDLE_ID'),
        404
      );

      return;
    }

    $strategy = $this->params->get('ads_txt_strategy', 'file');

    switch ($strategy) {
      case 'file':
        $this->serveFromFile($bundleId);
        break;

      case 'cache':
        $this->serveFromCache($bundleId);
        break;

      case 'dynamic':
      default:
        $this->serveDynamic($bundleId);
        break;
    }
  }

  /**
   * Serves ads.txt from physical file with fallback to dynamic.
   *
   * @param   string  $bundleId  The bundle ID.
   *
   * @return  void
   *
   * @since   1.0.0
   */
  private function serveFromFile(string $bundleId): void
  {
    $filePath = JPATH_ROOT . '/ads.txt';

    if (file_exists($filePath) && is_readable($filePath)) {
      $content = file_get_contents($filePath);

      if ($content !== false && !empty($content)) {
        $this->sendPlainTextResponse($content);

        return;
      }
    }

    // Fallback to dynamic serving
    $this->serveDynamic($bundleId);
  }

  /**
   * Serves ads.txt from cache with fallback to dynamic fetch.
   *
   * @param   string  $bundleId  The bundle ID.
   *
   * @return  void
   *
   * @since   1.0.0
   */
  private function serveFromCache(string $bundleId): void
  {
    try {
      $cacheController = Factory::getContainer()
        ->get(CacheControllerFactoryInterface::class)
        ->createCacheController('output', ['defaultgroup' => 'plg_system_monetumo']);

      $cacheId = 'ads_txt_' . md5($bundleId);
      $content = $cacheController->get($cacheId);

      if ($content !== false && !empty($content)) {
        $this->sendPlainTextResponse($content);

        return;
      }

      // Cache miss - fetch and cache
      $content = $this->fetchRemoteAdsTxt($bundleId);

      if (!empty($content)) {
        $cacheController->store($content, $cacheId);
        $this->sendPlainTextResponse($content);

        return;
      }
    } catch (\Exception $e) {
      // Log error but continue to dynamic serving
      Factory::getApplication()->enqueueMessage(
        'Monetumo cache error: ' . $e->getMessage(),
        'warning'
      );
    }

    // Fallback to dynamic serving
    $this->serveDynamic($bundleId);
  }

  /**
   * Serves ads.txt by fetching from remote URL.
   *
   * @param   string  $bundleId  The bundle ID.
   *
   * @return  void
   *
   * @since   1.0.0
   */
  private function serveDynamic(string $bundleId): void
  {
    $content = $this->fetchRemoteAdsTxt($bundleId);

    if (!empty($content)) {
      $this->sendPlainTextResponse($content);

      return;
    }

    // If fetch fails, return error message instead of redirecting
    $this->sendPlainTextResponse(
      '# ads.txt temporarily unavailable',
      503
    );
  }

  /**
   * Fetches ads.txt content from remote Monetumo server.
   *
   * @param   string  $bundleId  The bundle ID.
   *
   * @return  string  The ads.txt content or empty string on failure.
   *
   * @since   1.0.0
   */
  private function fetchRemoteAdsTxt(string $bundleId): string
  {
    $remoteUrl = $this->getRemoteAdsTxtUrl($bundleId);

    if (!function_exists('curl_init')) {
      return '';
    }

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $remoteUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Joomla-Monetumo/1.0');

    $content = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);

    if ($httpCode === 200 && !empty($content) && empty($error)) {
      return $content;
    }

    return '';
  }

  /**
   * Constructs the remote ads.txt URL.
   *
   * @param   string  $bundleId  The bundle ID.
   *
   * @return  string  The remote URL.
   *
   * @since   1.0.0
   */
  private function getRemoteAdsTxtUrl(string $bundleId): string
  {
    return 'https://monetumo.com/ads-txt/' . $this->sanitizeBundleId($bundleId);
  }

  /**
   * Sends a plain text response and exits.
   *
   * @param   string   $content     The content to send.
   * @param   integer  $statusCode  HTTP status code (default 200).
   *
   * @return  void
   *
   * @since   1.0.0
   */
  private function sendPlainTextResponse(
    string $content,
    int $statusCode = 200
  ): void {
    // WordPress-style: Simple and direct output
    if ($statusCode !== 200) {
      http_response_code($statusCode);
    }
    
    header('Content-Type: text/plain; charset=utf-8');
    echo $content;
    exit;
  }

  /**
   * Sanitizes the bundle ID for safe use in URLs.
   *
   * @param   string  $bundleId  The raw bundle ID.
   *
   * @return  string  The sanitized bundle ID.
   *
   * @since   1.0.0
   */
  private function sanitizeBundleId(string $bundleId): string
  {
    return preg_replace('/[^A-Za-z0-9_\-]/', '', $bundleId);
  }
}
