<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  Task.monetumoadsscheduler
 *
 * @copyright   Copyright (C) 2025 Monetumo. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\Task\Monetumoadsscheduler\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Component\Scheduler\Administrator\Event\ExecuteTaskEvent;
use Joomla\Component\Scheduler\Administrator\Task\Status;
use Joomla\Component\Scheduler\Administrator\Traits\TaskPluginTrait;
use Joomla\Event\SubscriberInterface;
use Joomla\Registry\Registry;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Monetumo Scheduler Task Plugin
 * Handles scheduled ads.txt refresh from Monetumo server
 *
 * @since  1.0.0
 */
final class MonetumoScheduler extends CMSPlugin implements SubscriberInterface
{
  use TaskPluginTrait;

  /**
   * @var string[]
   * @since 1.0.0
   */
  private const TASKS_MAP = [
    'monetumo.refreshAdsTxt' => [
      'langConstPrefix' => 'PLG_TASK_MONETUMOADSSCHEDULER_REFRESH_ADS_TXT',
      'method'          => 'refreshAdsTxt',
    ],
  ];

  /**
   * @var boolean
   * @since 1.0.0
   */
  protected $autoloadLanguage = true;

  /**
   * Returns an array of events this subscriber will listen to.
   *
   * @return array
   *
   * @since  1.0.0
   */
  public static function getSubscribedEvents(): array
  {
    return [
      'onTaskOptionsList'    => 'advertiseRoutines',
      'onExecuteTask'        => 'standardRoutineHandler',
      'onContentPrepareForm' => 'enhanceTaskItemForm',
    ];
  }

  /**
   * Refresh ads.txt file from Monetumo server
   *
   * @param   ExecuteTaskEvent  $event  The execute task event
   *
   * @return  integer  The task status
   *
   * @since   1.0.0
   */
  private function refreshAdsTxt(ExecuteTaskEvent $event): int
  {
    // Get the system plugin settings
    $systemPlugin = PluginHelper::getPlugin('system', 'monetumoads');
    
    if (!$systemPlugin) {
      $this->logTask('Monetumo system plugin is not installed or enabled');
      return Status::KNOCKOUT;
    }

    $params = new Registry($systemPlugin->params);
    $bundleId = $params->get('bundle_id', '');
    $strategy = $params->get('ads_txt_strategy', 'file');

    if (empty($bundleId)) {
      $this->logTask('Bundle ID is not configured');
      return Status::KNOCKOUT;
    }

    // Fetch ads.txt content from Monetumo server
    $content = $this->fetchRemoteAdsTxt($bundleId);

    if (empty($content)) {
      $this->logTask('Failed to fetch ads.txt from remote server');
      return Status::KNOCKOUT;
    }

    // Write file if strategy is set to 'file'
    if ($strategy === 'file') {
      $filePath = JPATH_ROOT . '/ads.txt';
      $result = file_put_contents($filePath, $content);
      
      if ($result === false) {
        $this->logTask('Failed to write ads.txt file');
        return Status::KNOCKOUT;
      }
      
      $this->logTask('Successfully updated ads.txt (' . number_format($result) . ' bytes)');
    }

    return Status::OK;
  }

  /**
   * Fetch ads.txt content (exact copy from system plugin)
   */
  private function fetchRemoteAdsTxt(string $bundleId): string
  {
    $remoteUrl = 'https://monetumo.com/ads-txt/' . preg_replace('/[^A-Za-z0-9_\-]/', '', $bundleId);

    if (!function_exists('curl_init')) {
      return '';
    }

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $remoteUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Joomla-Monetumo/1.0');

    $content = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);

    if ($httpCode === 200 && !empty($content) && empty($error)) {
      return $content;
    }

    return '';
  }

}
