<?php
/**
 * @package     Monetumo Ad Monetization
 * @subpackage  pkg_monetumo_ad_monetization
 * @copyright   Copyright (C) 2025 Monetumo. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

defined('_JEXEC') or die;

/**
 * Backward-compatible installer script that works on Joomla 3.x (to block installation)
 * and Joomla 5.x (to install properly)
 */
class pkg_monetumo_ad_monetizationInstallerScript
{
  private $minimumJoomla = '5.0';
  private $minimumPhp = '8.1';

  /**
   * Method to run before an install/update/uninstall method
   * Joomla 3.x, 4.x, and 5.x compatible
   */
  public function preflight($type, $parent)
  {
    // Check PHP version
    if (version_compare(PHP_VERSION, $this->minimumPhp, '<')) {
      $this->showError(
        'This package requires PHP ' . $this->minimumPhp . ' or higher. '
        . 'You are running PHP ' . PHP_VERSION . '.'
      );
      return false;
    }

    // Check Joomla version
    if (version_compare(JVERSION, $this->minimumJoomla, '<')) {
      $this->showError(
        'This package requires Joomla ' . $this->minimumJoomla . ' or higher. '
        . 'You are running Joomla ' . JVERSION . '.'
      );
      return false;
    }

    return true;
  }

  /**
   * Method to install the component
   */
  public function install($parent)
  {
    return true;
  }

  /**
   * Method to update the component
   */
  public function update($parent)
  {
    return true;
  }

  /**
   * Method to uninstall the component
   */
  public function uninstall($parent)
  {
    return true;
  }

  /**
   * Method to run after an install/update method
   */
  public function postflight($type, $parent)
  {
    // Only run on Joomla 5.x (after version check passed)
    if ($type === 'install' || $type === 'update') {
      $this->enablePlugins();
    }

    if ($type === 'install') {
      $this->showMessage(
        'Monetumo Ad Monetization package has been successfully installed! '
        . 'Both plugins have been enabled automatically.',
        'message'
      );
      $this->showMessage(
        '<strong>Important:</strong> Please configure the Monetumo Ads plugin '
        . '(System → Plugins → System - Monetumo Ads) with your Bundle ID before use.',
        'notice'
      );
    } elseif ($type === 'update') {
      $this->showMessage(
        'Monetumo Ad Monetization package has been successfully updated!',
        'message'
      );
    }

    return true;
  }

  /**
   * Enable plugins after installation
   */
  private function enablePlugins()
  {
    // Use Joomla 5.x DI container if available
    if (class_exists('Joomla\CMS\Factory') && method_exists('Joomla\CMS\Factory', 'getContainer')) {
      try {
        $db = JFactory::getContainer()->get('Joomla\Database\DatabaseInterface');
      } catch (Exception $e) {
        $db = JFactory::getDbo();
      }
    } else {
      $db = JFactory::getDbo();
    }

    $plugins = [
      ['system', 'monetumoads'],
      ['task', 'monetumoadsscheduler']
    ];

    foreach ($plugins as $plugin) {
      try {
        $query = $db->getQuery(true)
          ->update($db->quoteName('#__extensions'))
          ->set($db->quoteName('enabled') . ' = 1')
          ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'))
          ->where($db->quoteName('folder') . ' = ' . $db->quote($plugin[0]))
          ->where($db->quoteName('element') . ' = ' . $db->quote($plugin[1]));

        $db->setQuery($query);
        $db->execute();
      } catch (Exception $e) {
        // Silently fail - not critical
      }
    }
  }

  /**
   * Show error message (compatible with Joomla 3.x and 5.x)
   */
  private function showError($message)
  {
    if (class_exists('JFactory')) {
      JFactory::getApplication()->enqueueMessage($message, 'error');
    }
  }

  /**
   * Show message (compatible with Joomla 3.x and 5.x)
   */
  private function showMessage($message, $type = 'message')
  {
    if (class_exists('JFactory')) {
      JFactory::getApplication()->enqueueMessage($message, $type);
    }
  }
}
